function [out,fmin,parout] =nested_fista(PhiFun,Phifun_prox,Ffun,Ffun_grad,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,par)
%NESTED_FISTA employs the nested fista method for solving the problem min{phi(f(x)) + lambda* g(A*x)}
%
% Underlying assumptions:
% All functions are convex
% phi is Lipschitz, nondecreasing and proximable
% f =(f1,f2,...,fm). The components f1,f2,...,fm are smooth
% g is proper closed and proximable
% A is a linear transformation
% lambda is a positive scalar
% ====================================================
% Usage:
% out               = NESTED_FISTA(PhiFun,Phifun_prox,Ffun,Ffun_grad,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[par])
% [out,fmin]        = NESTED_FISTA(PhiFun,Phifun_prox,Ffun,Ffun_grad,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[par])
% [out,fmin,parout] = NESTED_FISTA(PhiFun,Phifun_prox,Ffun,Ffun_grad,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[par])
% ====================================================
% Input:
% PhiFun      - function handle for the function phi
% Phifun_prox - function handle for the proximal mapping of phi times a postive constant
% Ffun        - function handle for the function f (returns an m-dimensional column vector)
% Ffun_grad   - function handle for the transposed Jacobian of f 
% Gfun        - function handle for the function g
% Gfun_prox   - function handle for the proximal mapping of g times a postive constant
% Afun        - function handle for the linear transformation of A
% Atfun       - function handle for the adjoint of the linear transformation of A
% lambda      - positive scalar penalty for the function g
% startx      - starting vector
% par         - struct which contains different values required for the operation of NESTED-FISTA
% Fields of par:
%       max_iter       - maximal number of iterations [default: 1000]
%       print_flag     - true if internal printing should take place, otherwise false [default: true]
%       Lstart         - initial value for the outer Lipschitz constant (scalar) [default: 1]
%       eta            - the multiplicative constant used when regretting or backtracking [default: 2]
%       stuck_iter     - maximal allowed number of iterations without improvement [default: 100]
%       inner_eps      - stopping criteria tolerance for the inner fdpg method [default: 1e-5]
%       inner_max_iter - maximal number of iterations for the inner fdpg method [default: 50]
%       inner_Lstart   - initial" value for the Lipschitz constant for the inner fdpg method [default: 1]
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    LValVec   - vector of all used outer Lipschitz constants 

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

if ((nargin < 10) || ...
        ~isa(PhiFun,'function_handle') || ~isa(Phifun_prox,'function_handle') || ~isa(Ffun,'function_handle') || ...
        ~isa(Ffun_grad,'function_handle') || ~isa(Gfun,'function_handle') || ~isa(Gfun_prox,'function_handle') ||...
        ~isa(Afun,'function_handle') || ~isa(Atfun,'function_handle')  )
    error ('usage: nested_fista(PhiFun,Phifun_prox,Ffun,Ffun_grad,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[par]) ') ;
end

%setting default values

if (nargin < 11)
    par = struct() ;
end

    par = initialize_par_for_nested_fista(par) ;


%constant values
Lstart = par.Lstart ;
max_iter = par.max_iter ;
print_flag = par.print_flag ;
stuck_iter = par.stuck_iter ;
eta = par.eta ;


if (nargout == 3)
    parout = struct ;
    parout.iterNum = 0 ;
    parout.funValVec = [] ;
    parout.LValVec = [] ;
end


Feps = 1e-10 ; %Inner epsilon for comparing newFvalue and quad_aprrox

%starting values for the outer problem
iternu=0; %iteration number
prevx = startx ;
newx = prevx ;
newy = startx ;
prevt = 1 ;
startFval = Ffun(startx) ;
Astartval = Afun(startx) ;
prevFGvalue = PhiFun(startFval) + Gfun(Astartval) ;
Lout = Lstart ;
mlen = size(startFval,1) ; 
plen = length(Astartval) ;

savediter= -1 ; %last iter where a real change was performed 
done = false ;

%starting values for the inner problem
inpar.max_iter = par.inner_max_iter ;
inpar.Lstart = par.inner_Lstart ;
inpar.eps = par.inner_eps ;
inpar.print_flag = false;
inpar.feas_check_flag = false ;
instarty = zeros(mlen+plen,1) ;

%First printing
if(print_flag)
    fprintf('*********************\n');
    fprintf('nested-fista\n') ;
    fprintf('*********************\n')
    fprintf('#iter       \tfun. val.   \t L val. \t inner L val. \t inner iternu. \n');
end

%the iterations
while((iternu < max_iter) && ~done)
    iternu = iternu + 1;

    gradFy = Ffun_grad(newy) ;
    valFy = Ffun(newy) ;
    valGradFy = valFy - gradFy' * newy   ;
    %inA = [ gradFy' ; A] ;
    %inAt = inA' ;
    
    [outfdpg,~,backpar] = fdpg( @(x) inFfun(x),@(x) inConjfun(x),@(x) inGfun(x),@(x,lamda) inGProx(x,lamda),@(x) inAfun(x) ,@(x) inAtfun(x),1,instarty,inpar) ;
    
    inpar.Lstart = backpar.LValVec(end) / eta ;
    
    instarty = backpar.dualVec ;
    PhiF_outfdpg = PhiFun(Ffun(outfdpg));
    
    while (PhiF_outfdpg > (PhiFun(valFy + gradFy'*(outfdpg- newy)) + 0.5*Lout*norm(outfdpg- newy,2)^2  + Feps))
        Lout = Lout * eta ;
        [outfdpg,~,backpar] = fdpg( @(x) inFfun(x),@(x) inConjfun(x),@(x) inGfun(x),@(x,lamda) inGProx(x,lamda),@(x) inAfun(x) ,@(x) inAtfun(x),1,instarty,inpar) ;
        inpar.Lstart = backpar.LValVec(end) /eta;
        instarty = backpar.dualVec ;
        PhiF_outfdpg = PhiFun(Ffun(outfdpg));
    end
    
    newz = outfdpg ;
    valzk = PhiF_outfdpg  + lambda * Gfun(Afun(newz)) ;
    if (prevFGvalue < valzk)
        newx = prevx ;
        newFGvalue = prevFGvalue ;
    else
        newx = newz;
        newFGvalue = valzk ;
        savediter = iternu ;
    end
        
    
    newt = 0.5* (1+sqrt(1+4*prevt^2)) ;
    newy = newx + prevt /newt *(newz-newx) + (prevt -1)/newt *(newx-prevx) ;
    
    if (print_flag)
        fprintf('%6d \t    %12f \t %12f \t %6d  \t %6d \n',iternu,newFGvalue,Lout,inpar.Lstart,backpar.iterNum);
    end
    
    if (iternu - savediter > stuck_iter)
        fprintf('Stopping because of %d iterations with no improvement\n',stuck_iter) ;
        done = true ;
    end
    
    if (nargout == 3)
        %updating parout
        parout.funValVec = [ parout.funValVec ; newFGvalue] ;
        parout.LValVec = [parout.LValVec ; Lout] ;
    end
    
    prevx = newx ;
    prevt = newt ;
    prevFGvalue = newFGvalue ;
    
end

out = newx ;

if ((nargout >1) || (print_flag))
    fmin = newFGvalue ;
end

if (nargout == 3)
    %updating parout.iterNum
    parout.iterNum = iternu ;
end

if (print_flag)
    fprintf('----------------------------------\n') ;
    fprintf('Optimal value = %15f \n',fmin) ;
end

  
    function val = inFfun(x)
        val = 0.5 * Lout*norm(x-newy,2)^2 ;
    end

    function val = inGfun(x)
        x1 = x(1:mlen) ;
        x2 = x(mlen+1:mlen+plen) ;
        val =  PhiFun(valGradFy + x1) + lambda * Gfun(x2) ;
    end

    function valprox = inGProx(x,alpha)
        x1 = x(1:mlen) ;
        x2 = x(mlen+1:mlen+plen) ;
        valprox = [Phifun_prox(valGradFy + x1,alpha) - valGradFy; Gfun_prox(x2,alpha*lambda)] ;
    end

    function valcong = inConjfun(w)
        valcong = w/Lout + newy ;
    end

    function valAf = inAfun(x)
        valAf = [ gradFy'*x ; Afun(x)] ;
    end

    function valAtf = inAtfun(x)
        x1 = x(1:mlen) ;
        x2 = x(mlen+1:mlen+plen) ;
        valAtf =  gradFy*x1 + Atfun(x2) ;
    end

end


